<?php
// app/services/BinaryService.php

final class BinaryService
{
    /**
     * Place member under $parentId on leg L/R.
     * If that leg already filled, cari slot kosong ke bawah (BFS).
     */
    public static function place(PDO $pdo, int $memberId, int $parentId, string $leg): array
    {
        if (!in_array($leg, ['L','R'], true)) {
            throw new Exception("Leg harus 'L' atau 'R'");
        }

        // pastikan member belum ada di binary_tree
        $st = $pdo->prepare("SELECT member_id FROM binary_tree WHERE member_id=? LIMIT 1");
        $st->execute([$memberId]);
        if ($st->fetch()) {
            throw new Exception("Member sudah ada di binary_tree");
        }

        // cari parent final (slot kosong) untuk leg ini
        $finalParent = self::findAvailableParent($pdo, $parentId, $leg);

        // insert ke binary_tree
        $pdo->prepare("INSERT INTO binary_tree (member_id, parent_id, leg) VALUES (?,?,?)")
            ->execute([$memberId, $finalParent, $leg]);

        // build closure rows
        self::buildClosure($pdo, $memberId, $finalParent, $leg);

        return ['parent_id' => $finalParent, 'leg' => $leg];
    }

    /**
     * BFS cari node yang punya slot kosong di leg tertentu.
     */
    private static function findAvailableParent(PDO $pdo, int $startParentId, string $leg): int
    {
        // jika startParentId langsung kosong di leg itu -> pakai
        if (!self::isLegTaken($pdo, $startParentId, $leg)) {
            return $startParentId;
        }

        // BFS: jelajah anak kiri/kanan untuk cari parent yang kosong
        $queue = [$startParentId];
        while (!empty($queue)) {
            $p = array_shift($queue);

            // ambil children p
            $st = $pdo->prepare("SELECT member_id FROM binary_tree WHERE parent_id=?");
            $st->execute([$p]);
            $children = $st->fetchAll();

            foreach ($children as $c) {
                $queue[] = (int)$c['member_id'];
            }

            // cek apakah node ini punya slot kosong di leg yang diminta
            if (!self::isLegTaken($pdo, $p, $leg)) {
                return $p;
            }
        }

        throw new Exception("Tidak menemukan slot kosong (struktur rusak?)");
    }

    private static function isLegTaken(PDO $pdo, int $parentId, string $leg): bool
    {
        $st = $pdo->prepare("SELECT 1 FROM binary_tree WHERE parent_id=? AND leg=? LIMIT 1");
        $st->execute([$parentId, $leg]);
        return (bool)$st->fetchColumn();
    }

    /**
     * Build binary_closure for new member:
     * - self row (member->member depth 0)
     * - for each ancestor of parent: add ancestor->member depth+1 and first_leg inherited
     * - add parent->member depth 1 with first_leg = $leg (karena langkah pertama dari parent ke child)
     */
    private static function buildClosure(PDO $pdo, int $memberId, int $parentId, string $leg): void
    {
        // 1) self
        $pdo->prepare("
            INSERT IGNORE INTO binary_closure (ancestor_id, descendant_id, depth, first_leg)
            VALUES (?, ?, 0, NULL)
        ")->execute([$memberId, $memberId]);

        // 2) parent -> member (depth 1), first_leg dari parent ke member = $leg
        $pdo->prepare("
            INSERT IGNORE INTO binary_closure (ancestor_id, descendant_id, depth, first_leg)
            VALUES (?, ?, 1, ?)
        ")->execute([$parentId, $memberId, $leg]);

        // 3) semua ancestor dari parent
        $st = $pdo->prepare("
            SELECT ancestor_id, depth, first_leg
            FROM binary_closure
            WHERE descendant_id = ?
        ");
        $st->execute([$parentId]);
        $ancestors = $st->fetchAll();

        foreach ($ancestors as $a) {
            $ancestorId = (int)$a['ancestor_id'];
            $depth = (int)$a['depth'] + 1;

            // first_leg untuk ancestor->member = first_leg ancestor->parent (bukan $leg),
            // karena langkah pertama dari ancestor menuju parent sudah menentukan sisi L/R.
            $firstLeg = $a['first_leg'];
            // Untuk kasus ancestor==parent, first_leg di atas mungkin NULL (depth=0),
            // tapi parent->member sudah kita insert di step (2), jadi aman.
            if ($ancestorId === $parentId) continue;

            $pdo->prepare("
                INSERT IGNORE INTO binary_closure (ancestor_id, descendant_id, depth, first_leg)
                VALUES (?, ?, ?, ?)
            ")->execute([$ancestorId, $memberId, $depth, $firstLeg]);
        }
    }
}
