<?php
$pdo = Db::pdo();
$myRootId = (int)$member['id'];
$MAX_DEPTH = 4;

// =====================
// LOAD ALL PLACEMENTS (simple for dev)
// =====================
$placements = $pdo->query("SELECT member_id, parent_id, leg FROM binary_placements")->fetchAll(PDO::FETCH_ASSOC);

// maps
$childOf = [];          // parent_id => ['L'=>member_id, 'R'=>member_id]
$parentOf = [];         // member_id => ['parent_id'=>..., 'leg'=>...]
$allIds = [$myRootId => true];

foreach ($placements as $r) {
  $m = (int)$r['member_id'];
  $p = (int)$r['parent_id'];
  $leg = strtoupper(trim($r['leg']));

  if ($leg !== 'L' && $leg !== 'R') continue;

  if (!isset($childOf[$p])) $childOf[$p] = ['L'=>0,'R'=>0];
  $childOf[$p][$leg] = $m;

  $parentOf[$m] = ['parent_id'=>$p,'leg'=>$leg];

  $allIds[$m] = true;
  $allIds[$p] = true;
}

// =====================
// HELPER: check descendant (only allow drill within my subtree)
// =====================
function isDescendant(array $parentOf, int $myRootId, int $targetId): bool {
  if ($targetId === $myRootId) return true;
  $cur = $targetId;
  $guard = 0;
  while (isset($parentOf[$cur])) {
    $cur = (int)$parentOf[$cur]['parent_id'];
    if ($cur === $myRootId) return true;
    $guard++;
    if ($guard > 50000) break; // safety
  }
  return false;
}

// requested drill root
$requestedRoot = isset($_GET['root']) ? (int)$_GET['root'] : $myRootId;
$currentRootId = $requestedRoot > 0 && isDescendant($parentOf, $myRootId, $requestedRoot) ? $requestedRoot : $myRootId;

// =====================
// MEMBER INFO CACHE
// =====================
function loadMembers(PDO $pdo, array $ids): array {
  $ids = array_values(array_unique(array_filter($ids, function($x){ return (int)$x > 0; })));
  if (empty($ids)) return [];

  // build placeholders
  $in = implode(',', array_fill(0, count($ids), '?'));
  $st = $pdo->prepare("
    SELECT m.id, m.username, m.status, p.code AS package_code
    FROM members m
    LEFT JOIN packages p ON p.id = m.package_id
    WHERE m.id IN ($in)
  ");
  $st->execute($ids);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);

  $map = [];
  foreach ($rows as $r) {
    $map[(int)$r['id']] = [
      'id' => (int)$r['id'],
      'username' => $r['username'],
      'status' => $r['status'],
      'package_code' => $r['package_code'] ?: '-',
    ];
  }
  return $map;
}

// =====================
// SUBTREE COUNTS (memoized)
// count descendants for a node (including all levels)
// =====================
$memo = []; // nodeId => countDescendants
function countDesc(array $childOf, int $nodeId, array &$memo): int {
  if ($nodeId <= 0) return 0;
  if (isset($memo[$nodeId])) return $memo[$nodeId];

  $left = $childOf[$nodeId]['L'] ?? 0;
  $right = $childOf[$nodeId]['R'] ?? 0;

  $cnt = 0;
  if ($left > 0)  $cnt += 1 + countDesc($childOf, $left, $memo);
  if ($right > 0) $cnt += 1 + countDesc($childOf, $right, $memo);

  $memo[$nodeId] = $cnt;
  return $cnt;
}

function nodeBadge(string $pkg): string {
  $pkg = strtoupper($pkg);
  if ($pkg === 'WARRIOR') return 'bg-secondary';
  if ($pkg === 'SMART')   return 'bg-primary';
  if ($pkg === 'RICH')    return 'bg-warning text-dark';
  return 'bg-dark';
}

// =====================
// BUILD TREE LEVELS
// =====================
function buildTreeLevels(array $childOf, int $rootId, int $maxDepth): array {
  $levels = [];
  $levels[0] = [['id'=>$rootId]];

  for ($d=1; $d<$maxDepth; $d++) {
    $prev = $levels[$d-1];
    $cur = [];
    foreach ($prev as $n) {
      $id = (int)$n['id'];
      if ($id > 0) {
        $l = $childOf[$id]['L'] ?? 0;
        $r = $childOf[$id]['R'] ?? 0;
        $cur[] = ['id'=>(int)$l];
        $cur[] = ['id'=>(int)$r];
      } else {
        $cur[] = ['id'=>0];
        $cur[] = ['id'=>0];
      }
    }
    $levels[$d] = $cur;
  }
  return $levels;
}

$levels = buildTreeLevels($childOf, $currentRootId, $MAX_DEPTH);

// collect ids needed for display + their children (for counts display)
$needIds = [];
foreach ($levels as $lv) foreach ($lv as $n) if ((int)$n['id']>0) $needIds[] = (int)$n['id'];

// also include immediate children ids for correct count display on visible nodes (not necessary, but ok)
foreach ($needIds as $nid) {
  $needIds[] = $childOf[$nid]['L'] ?? 0;
  $needIds[] = $childOf[$nid]['R'] ?? 0;
}
$needIds[] = $myRootId;

$membersMap = loadMembers($pdo, $needIds);

// breadcrumb: from current root back up to myRootId
$crumb = [];
$cur = $currentRootId;
$guard = 0;
while (true) {
  $crumb[] = $cur;
  if ($cur === $myRootId) break;
  if (!isset($parentOf[$cur])) break;
  $cur = (int)$parentOf[$cur]['parent_id'];
  $guard++;
  if ($guard > 2000) break;
}
$crumb = array_reverse($crumb);

// url helper
function network_url(int $root): string {
  return '/jwdnetwork/member.php?page=network&root='.(int)$root;
}
?>

<style>
.tree-wrap{ overflow-x:auto; padding-bottom:8px; }
.tree-level{ display:flex; justify-content:center; gap:18px; margin:18px 0; min-width:980px; }
.tree-card{
  width:200px; background:#fff; border:1px solid #e9ecef;
  border-radius:14px; padding:10px 12px;
  box-shadow:0 6px 18px rgba(0,0,0,.04);
}
.tree-card.empty{ background:#f8f9fa; border-style:dashed; }
.tree-card a{ text-decoration:none; color:inherit; display:block; }
.tree-card:hover{ border-color:#cfd4da; }
.meta-row{ display:flex; justify-content:space-between; gap:8px; }
.meta-row .small{ line-height:1.2; }
</style>

<div class="row g-3">
  <div class="col-12">
    <div class="card border-0 shadow-sm">
      <div class="card-body d-flex flex-wrap justify-content-between align-items-center gap-2">
        <div>
          <h5 class="mb-1">Binary Network</h5>
          <div class="text-secondary small">
            Klik node untuk drill-down. Max <?= (int)$MAX_DEPTH ?> level.
          </div>
        </div>
        <div class="small">
          <span class="badge bg-secondary">WARRIOR</span>
          <span class="badge bg-primary">SMART</span>
          <span class="badge bg-warning text-dark">RICH</span>
        </div>
      </div>
    </div>
  </div>

  <!-- Breadcrumb -->
  <div class="col-12">
    <div class="card border-0 shadow-sm">
      <div class="card-body d-flex flex-wrap justify-content-between align-items-center gap-2">
        <div class="small">
          <span class="text-secondary">Path:</span>
          <?php foreach ($crumb as $i => $id): ?>
            <?php
              $u = $membersMap[$id]['username'] ?? ('#'.$id);
              $isLast = ($i === count($crumb)-1);
            ?>
            <?php if (!$isLast): ?>
              <a href="<?= network_url($id) ?>" class="text-decoration-none"><?= e($u) ?></a>
              <span class="text-secondary">/</span>
            <?php else: ?>
              <span class="fw-semibold"><?= e($u) ?></span>
            <?php endif; ?>
          <?php endforeach; ?>
        </div>
        <div class="d-flex gap-2">
          <?php if ($currentRootId !== $myRootId): ?>
            <a class="btn btn-outline-secondary btn-sm" href="<?= network_url($myRootId) ?>">
              <i class="bi bi-house me-1"></i> My Root
            </a>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>

  <!-- Tree -->
  <div class="col-12">
    <div class="card border-0 shadow-sm">
      <div class="card-body">
        <div class="tree-wrap">
          <?php foreach ($levels as $lv): ?>
            <div class="tree-level">
              <?php foreach ($lv as $n): ?>
                <?php
                  $id = (int)$n['id'];
                  if ($id <= 0) {
                    echo '<div class="tree-card empty"><div class="small text-secondary">EMPTY</div><div class="fw-semibold">-</div></div>';
                    continue;
                  }

                  $info = $membersMap[$id] ?? ['username'=>'#'.$id,'status'=>'-','package_code'=>'-','id'=>$id];
                  $pkg = $info['package_code'] ?? '-';
                  $badge = nodeBadge($pkg);
                  $statusClass = (($info['status'] ?? '') === 'active') ? 'text-success' : 'text-secondary';

                  // counts
                  $leftId  = $childOf[$id]['L'] ?? 0;
                  $rightId = $childOf[$id]['R'] ?? 0;
                  $leftCnt  = $leftId  ? (1 + countDesc($childOf, (int)$leftId, $memo)) : 0;
                  $rightCnt = $rightId ? (1 + countDesc($childOf, (int)$rightId, $memo)) : 0;
                ?>
                <div class="tree-card">
                  <a href="<?= network_url($id) ?>">
                    <div class="d-flex justify-content-between align-items-center mb-1">
                      <span class="badge <?= $badge ?>"><?= e($pkg) ?></span>
                      <span class="small <?= $statusClass ?>"><?= e(strtoupper($info['status'] ?? '-')) ?></span>
                    </div>

                    <div class="fw-bold"><?= e($info['username'] ?? '') ?></div>
                    <div class="small text-secondary">ID: <?= (int)$id ?></div>

                    <div class="meta-row mt-2">
                      <div class="small text-secondary">Left<br><b class="text-dark"><?= (int)$leftCnt ?></b></div>
                      <div class="small text-secondary text-end">Right<br><b class="text-dark"><?= (int)$rightCnt ?></b></div>
                    </div>
                  </a>
                </div>
              <?php endforeach; ?>
            </div>
          <?php endforeach; ?>
        </div>

        <div class="text-secondary small mt-2">
          Total Left/Right = jumlah downline di kaki kiri/kanan dari node tersebut (semua level).
        </div>
      </div>
    </div>
  </div>
</div>
